using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.Common.Log;              //251417 - for logging of any kind
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;     //251417 - for locking failure email
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

namespace gov.va.med.vbecs.DAL.HL7AL
{

		#region Header

		//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		//<Developers>
		//	<Developer>Russell Stephenson</Developer>
		//</Developers>
		//<SiteName>Hines OIFO</SiteName>
		//<CreationDate>1/16/2016</CreationDate>
		//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		//<summary>This class provides support for Inbound HL7 messages coming from Automated Instruments into VBECS.</summary>

		#endregion

    /// <summary>
    /// PendingResultsControler
    /// </summary>
    public class PendingResultsController
    {
        // Events Logger - 251417 added for locking , plus, are we logging AI like the other interfaces?
        private static readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

        //Defines the minimum user id length.
        private static readonly int MIN_USER_ID_LENGTH = 9;

        //PT_114.02 PT_114.03
        //Provides mapping for test results to validate inbound data.
        //3 -> 3
        private static Hashtable _testResultLookup;
        private static Hashtable TestResultLookup
        {
            get
            {
                if (_testResultLookup == null)
                {
                    _testResultLookup = new Hashtable();

                    // Empty results
                    _testResultLookup.Add(string.Empty, "X");
                    _testResultLookup.Add("X", "X");

                    // Positive results
                    _testResultLookup.Add("1", "1");
                    _testResultLookup.Add("2", "2");
                    _testResultLookup.Add("3", "3");
                    _testResultLookup.Add("4", "4");
 
  
                    // Negative results
                    _testResultLookup.Add("0", "0");

                }

                return _testResultLookup;
            }
        }

        //Provides mapping for test interpritations to validate inbound data.
        //A -> A
        private static Hashtable _interpLookup;
        private static Hashtable InterpLookup
        {
            get
            {
                if (_interpLookup == null)
                {
                    _interpLookup = new Hashtable();

                    // Empty results
                    _interpLookup.Add(string.Empty, "X");
                    _interpLookup.Add("X", "X");

                    // Positive results
                    _interpLookup.Add("P", "P");
                    _interpLookup.Add("Pos", "P");
                    _interpLookup.Add("Positive", "P");

                    _interpLookup.Add("A", "A");
                    _interpLookup.Add("B", "B");
                    _interpLookup.Add("AB", "AB");
                    _interpLookup.Add("O", "O");

                    // Negative results
                    _interpLookup.Add("N", "N");
                    _interpLookup.Add("Neg", "N");
                    _interpLookup.Add("Negative", "N");
                }

                return _interpLookup;
            }
        }

        //BR 41.06
        //Provides mapping for test antigen interpritaions to when we add an interpritaion for each inbound result.
        //1 -> P
        private static Hashtable _antigenLookup;
        private static Hashtable AntigenLookUp
        {
            get
            {
                if (_antigenLookup == null)
                {
                    _antigenLookup = new Hashtable();

                    // Positive results
                    _antigenLookup.Add("1", "P");
                    _antigenLookup.Add("2", "P");
                    _antigenLookup.Add("3", "P");
                    _antigenLookup.Add("4", "P");
                    _antigenLookup.Add("0", "N");
                }

                return _antigenLookup;
            }
        }

        //Provides mapping for test crossmatch interpritaions for inbound data.
        //Comp -> C
        private static Hashtable _xminterpLookup;
        private static Hashtable XMInterpLookup
        {
            get
            {
                if (_xminterpLookup == null)
                {
                    _xminterpLookup = new Hashtable();

                    // Empty results
                    _xminterpLookup.Add(string.Empty, "X");
                    _xminterpLookup.Add("X", "X");

                    // Positive results
                    _xminterpLookup.Add("C", "C");
                    _xminterpLookup.Add("Comp", "C");
                    _xminterpLookup.Add("Compatible", "C");

                    _xminterpLookup.Add("I", "I");
                    _xminterpLookup.Add("Incomp", "I");
                    _xminterpLookup.Add("Incompatible", "I");
                }

                return _xminterpLookup;
            }
        }

        //Provides mapping for test the combined crossmatch interpritaions 
        //from ISInterp and AHGInterp to XMInterp for inbound data.
        //CC->C
        private static Hashtable _xminterpcombinedLookup;
        private static Hashtable XMInterpCombinedLookup
        {
            get
            {
                if (_xminterpcombinedLookup == null)
                {
                    _xminterpcombinedLookup = new Hashtable();

                    // Empty results
                    _xminterpcombinedLookup.Add(string.Empty, "X");
                    _xminterpcombinedLookup.Add("X", "X");

                    //Combined Interps
                    _xminterpcombinedLookup.Add("CC", "C");
                    _xminterpcombinedLookup.Add("CI", "I");
                    _xminterpcombinedLookup.Add("IC", "I");
                    _xminterpcombinedLookup.Add("II", "I");
                }

                return _xminterpcombinedLookup;
            }
        }

        /// <Developers>
        /// 	<Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        ///  <summary>
        /// Used for BR_114.04
        /// Gets the pending blood unit test if there is one that matches given parameters.
        /// There should never be more than one pending test for a given SpecimenUid.
        /// unitId = W696900000045 , divsionCode = 589, pendingTestStatusId = PendingTestStatus.Pending, BloodTestTypeId = positive number.
        ///  </summary>
        ///  <param name="unitId"> Blood unit ID</param>
        ///  <param name="divsionCode">Division Code</param>
        /// <param name="pendingTestStatusId">Is set to 1 if the test is still pending</param>
        /// <param name="bloodTestTypeId"></param>
        /// <returns><see cref="DataTable"/>containing pending blood unit test.</returns>
        public static DataTable GetPendingBloodUnitTest(string unitId, string divsionCode, PendingTestStatus pendingTestStatusId, int bloodTestTypeId)
        {
            DataSet ds;

            SqlParameter[] prms =
            {
                new SqlParameter(STOREDPROC.HL7GetPendingBloodUnitTest.BloodUnitId, SqlDbType.VarChar),
                new SqlParameter(STOREDPROC.HL7GetPendingBloodUnitTest.DivisionCode, SqlDbType.VarChar),
                new SqlParameter(STOREDPROC.HL7GetPendingBloodUnitTest.PendingTestStatusId, SqlDbType.Int), 
                new SqlParameter(STOREDPROC.HL7GetPendingBloodUnitTest.BloodTestTypeId, SqlDbType.Int), 
            };
            prms[0].Value = unitId;
            prms[1].Value = divsionCode;
            prms[2].Value = (int)pendingTestStatusId;
            prms[3].Value = bloodTestTypeId;
            
            try
            {
                ds = StoredProcedure.GetData(STOREDPROC.HL7GetPendingBloodUnitTest.StoredProcName, prms);
            }
            catch (Exception ex)
            {
                throw new DataAccessLayerException(String.Format("Internal error: can't get pending blood unit test data from VBECS DB for UnitId: {0}, DivsionCode: {1}, PendingTestStatusId {2} and BloodTestTypId: {3}", unitId, divsionCode, pendingTestStatusId.ToString(), bloodTestTypeId), ex);
            }
            return ds.Tables[0];
        }

        /// <Developers>
        /// 	<Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// BR_114.04
        /// Returns the pending specimen tests for the given parameters
        /// There should never be more than one pending test for a given SpecimenUid.
        /// </summary>
        /// <param name="specimenUid"> Specimen ID</param>
        /// <param name="divsionCode">Division Code</param>
        /// <param name="pendingTestStatusId">Pending Test Status ID is set to pending review if it is pended</param>
        /// <param name="bloodTestTypeId">Blood Test Type ID is used to verify this is the correct test.</param>
        /// <returns><see cref="DataTable"/> containing the pending specimen test.</returns>
        public static DataTable GetPendingSpecimenTest(string specimenUid, string divsionCode, PendingTestStatus pendingTestStatusId, int bloodTestTypeId)
        {
            DataSet ds;

            SqlParameter[] prms =
            {
                new SqlParameter(STOREDPROC.HL7GetPendingSpecimenTest.SpecimenUid, SqlDbType.VarChar),
                new SqlParameter(STOREDPROC.HL7GetPendingSpecimenTest.DivisionCode, SqlDbType.VarChar),
                new SqlParameter(STOREDPROC.HL7GetPendingSpecimenTest.PendingTestStatusId, SqlDbType.Int), 
                new SqlParameter(STOREDPROC.HL7GetPendingSpecimenTest.BloodTestTypeId, SqlDbType.Int), 
            };
            prms[0].Value = specimenUid;
            prms[1].Value = divsionCode;
            prms[2].Value = (int)pendingTestStatusId;
            prms[3].Value = bloodTestTypeId;

            try
            {
                ds = StoredProcedure.GetData(STOREDPROC.HL7GetPendingSpecimenTest.StoredProcName, prms);
            }
            catch (Exception ex)
            {
                throw new DataAccessLayerException(String.Format("Internal error: Internal error: can't get pending specimen test data from VBECS DB for SpecimenUid: {0}, DivsionCode: {1}, PendingTestStatusId {2} and BloodTestTypId: {3}", specimenUid, divsionCode, pendingTestStatusId, bloodTestTypeId), ex);
            }
            return ds.Tables[0];
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/16/2016</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        /// <summary>
        /// Facilitates BR_114.01 
        ///	If the blood unit record is not found, the message is rejected.
        /// This procedure returns the Blood Unit if it exists for the given parameters.
        /// There may be more than one row returned for a given UnitId because there maybe multiple products with this same UnitId
        /// </summary>
        /// <param name="unitId"> Blood unit ID</param>
        /// <param name="divsionCode">Division Code</param>
        /// <returns><see cref="DataTable"/> containing the Blood Unit if it exists.</returns>
        public static DataTable GetBloodUnit(string unitId, string divsionCode)
        {
            DataSet ds;

            SqlParameter[] prms =
            {
                new SqlParameter(STOREDPROC.GetBloodUnitsByUnitId.unitId, SqlDbType.VarChar),
                new SqlParameter(STOREDPROC.GetBloodUnitsByUnitId.divisionCode, SqlDbType.VarChar),
            };
            prms[0].Value = unitId;
            prms[1].Value = divsionCode;

            try
            {
                ds = StoredProcedure.GetData(STOREDPROC.GetBloodUnitsByUnitId.StoredProcName, prms);
            }
            catch (Exception ex)
            {
                throw new DataAccessLayerException(String.Format("Internal error: can't get blood unit data from VBECS DB for UnitId: {0}, DivsionCode: {1}.", unitId, divsionCode), ex);
           }

            return ds.Tables[0];
        }

        /// <Developers>
        /// 	<Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        ///  <summary>
        ///  Facilitates BR_114.02
        /// 	Returns specimen and order information for the given parameters.
        ///  </summary>
        ///  <param name="specimenUid">Specimen UID</param>
        ///  <param name="testPanelName">Division Code</param>
        /// <param name="bloodTestName">Used for finding the blo</param>
        /// <param name="divisionCode">Division Code defines the VHA division ex. 589 is </param>
        /// <returns><see cref="DataTable"/> containing specimen and order information</returns>
        public static DataTable GetSpecimen(string specimenUid, string testPanelName, string bloodTestName, string divisionCode)
        {
            DataSet ds;

            SqlParameter[] prms =
            {
                new SqlParameter(STOREDPROC.HL7GetOrderedTestBySpecimenUIDAndTestName.SpecimenUid, SqlDbType.VarChar),
                new SqlParameter(STOREDPROC.HL7GetOrderedTestBySpecimenUIDAndTestName.TestPanelName, SqlDbType.VarChar),
                new SqlParameter(STOREDPROC.HL7GetOrderedTestBySpecimenUIDAndTestName.BloodTestName, SqlDbType.VarChar),
                new SqlParameter(STOREDPROC.HL7GetOrderedTestBySpecimenUIDAndTestName.DivisionCode, SqlDbType.VarChar)
            };

            prms[0].Value = specimenUid;
            prms[1].Value = testPanelName;
            prms[2].Value = (!string.IsNullOrEmpty(bloodTestName))
                ? prms[2].Value = bloodTestName
                : prms[2].Value = null;
            prms[3].Value = divisionCode;

            try
            {
                ds = StoredProcedure.GetData(STOREDPROC.HL7GetOrderedTestBySpecimenUIDAndTestName.StoredProcName, prms);
            }
            catch (Exception ex)
            {
                throw new DataAccessLayerException(String.Format("Internal error: can't get specimen data from VBECS DB for UpecimenUid: {0}, TestPanelName: {1}.", specimenUid, testPanelName), ex);
            }
            
            if (ds != null && ds.Tables.Count > 0)
                return ds.Tables[0];

            return new DataTable();
        }

        /// <Developers>
        /// 	<Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        ///  <summary>
        ///  Facilitates BR_114.02
        /// 	Returns Crossmatch data and order information for the given parameters.
        ///  </summary>
        ///  <param name="specimenUid">Specimen UID</param>
        /// <param name="bloodUnitId"></param>
        /// <param name="divisionCode"></param>
        /// <returns><see cref="DataTable"/> containing specimen and order information</returns>
        public static DataTable HL7GetXMBySpecimenUIDAndUnitID(string specimenUid, string bloodUnitId, string divisionCode)
        {
            DataSet ds;

            SqlParameter[] prms =
            {
                new SqlParameter(STOREDPROC.HL7GetXMBySpecimenUIDAndUnitID.SpecimenUid, SqlDbType.VarChar),
                new SqlParameter(STOREDPROC.HL7GetXMBySpecimenUIDAndUnitID.BloodUnitId, SqlDbType.VarChar),
                new SqlParameter(STOREDPROC.HL7GetXMBySpecimenUIDAndUnitID.DivisionCode, SqlDbType.VarChar)
            };

            prms[0].Value = specimenUid;
            prms[1].Value = bloodUnitId;
            prms[2].Value = divisionCode;

            try
            {
                ds = StoredProcedure.GetData(STOREDPROC.HL7GetXMBySpecimenUIDAndUnitID.StoredProcName, prms);
            }
            catch (Exception ex)
            {
                throw new DataAccessLayerException(String.Format("Internal error: can't get XM data from VBECS DB for UpecimenUid: {0}, BloodUnitId: {1}, DivisionCode {2}.", specimenUid, bloodUnitId, divisionCode), ex);
            }

            return ds.Tables[0];
        }


        /// <Developers>
        /// 	<Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        ///  <summary>
        ///  Get the VBECSUserId for a given user id even if the input user id has characters removed from the left side.
        ///  </summary>
        /// <param name="userId"></param>
        /// <param name="validationErrors"></param>
        /// <returns><see cref="DataRow"/> containing the VbecsUser record where the VBECSUserId matched userId.</returns>
        public static DataRow GetUserRecord(string userId, ref List<string> validationErrors)
        {
            //Defect 304968
            if (userId.Length < MIN_USER_ID_LENGTH)
            {
                validationErrors.Add(string.Format("Tech ID: {0} is required to be at least {1} long.", userId, MIN_USER_ID_LENGTH));
                return null; 
            }
            SqlParameter[] parms =
            {
                new SqlParameter(VbecsStoredProcs.HL7GetVbecsUserWithVariableLengthID.VbecsUserId, SqlDbType.VarChar),
                new SqlParameter(VbecsStoredProcs.HL7GetVbecsUserWithVariableLengthID.SearchAllUsers, SqlDbType.Bit)
            };

            parms[0].Value = userId;
            parms[1].Value = true;

            DataTable userIdTable;

            try
            {
                userIdTable = StoredProcedure.GetDataTable(VbecsStoredProcs.HL7GetVbecsUserWithVariableLengthID.StoredProcName, parms);
            }
            catch (Exception ex)
            {
                throw new DataAccessLayerException(String.Format("Internal error: can't get user data from VBECS DB for UserId: {0}, SearchAllUsers: {1}.", userId, 1), ex);
            }


            if (userIdTable == null || userIdTable.Rows.Count == 0)
                return null;

            // Defect 304968
            if(userIdTable.Rows.Count > 1)
                validationErrors.Add(string.Format("Tech ID: {0} is insufficiently unique.", userId));

            return userIdTable.Rows[0];     
        }

        /// <Developers>
        /// 	<Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        ///  <summary>
        ///  Gets the numeric BloodTestTypeId for the given input parameters.
        ///  Returns the TestTypId and the M
        ///  </summary>
        ///  <param name="testName"></param>
        ///  <param name="testType"></param>
        /// <param name="orderableTestId"></param>
        /// <param name="testTypeId"></param>
        /// <param name="mappedTypeId"></param>
        /// <param name="validationErrors"></param>
        /// <returns><see cref="bool"/></returns>
        public static bool GetTestTypeMapping(string testName, string testType, int orderableTestId, 
            out int testTypeId, out int mappedTypeId, ref List<string> validationErrors)
        {
            //testTypeId and mappedTypeId are intialized to because there is no instance of 0 in the BloodTestTypeMapping table.
           
            testTypeId = 0;
            mappedTypeId = 0;
            SqlParameter[] parms =
            {
                new SqlParameter(VbecsStoredProcs.HL7GetBloodTestTypeMapping.BloodTestPanelName, SqlDbType.VarChar),
                new SqlParameter(VbecsStoredProcs.HL7GetBloodTestTypeMapping.BloodTestName, SqlDbType.VarChar),
                new SqlParameter(VbecsStoredProcs.HL7GetBloodTestTypeMapping.OrderableTestId, SqlDbType.Int)
            };

            parms[0].Value = testName;
            parms[1].Value = testType;

            //ordetableTestId is the value that is used in the OrderableTest table to match tests to test types.  
            //It is ouly used for patient test mapping.
            if(orderableTestId != 0)
                parms[2].Value  = orderableTestId;
            else
                parms[2].Value = DBNull.Value;

            DataTable testTypeMapping;
            try
            {
                testTypeMapping = StoredProcedure.GetDataTable(VbecsStoredProcs.HL7GetBloodTestTypeMapping.StoredProcName, parms);
            }
            catch (Exception ex)
            {
                throw new DataAccessLayerException(String.Format("Internal error: can't get test type mapping from VBECS DBfor TestName: {0}, TestType: {1} and orderableTestId {3}.", testName, testType, testType ), ex);
            }

           

            if (testTypeMapping == null || testTypeMapping.Rows.Count == 0 || testTypeMapping.Rows[0].ItemArray.Length == 0)
            {
                validationErrors.Add(string.Format(
                        "PendingResultController.GetTestTypeMapping: Failed to find TestTypeId for TestName: {0} and TestType {1}.",
                        testName, testType));
                return false;
            }
            if (!int.TryParse(testTypeMapping.Rows[0][VbecsTables.BloodTestTypeMapping.BloodTestTypeId].ToString(), out testTypeId))
            {
                validationErrors.Add(string.Format("PendingResultController.GetTestTypeMapping: Cannot convert TestTypeId {0} to type int.",
                        testTypeMapping.Rows[0][VbecsTables.BloodTestTypeMapping.BloodTestTypeId].ToString()));
                return false;
            }
            if (testTypeMapping.Rows[0].ItemArray.Length < 1)
            {
                validationErrors.Add(string.Format("PendingResultController.GetTestTypeMapping: mapped type ID doesn't exist for test type id {0}.", testTypeId));
                return false;
            }

            if (testTypeMapping.Rows[0][VbecsTables.BloodTestTypeMapping.AntigenTestTypeId] != null && !string.IsNullOrEmpty(testTypeMapping.Rows[0][VbecsTables.BloodTestTypeMapping.AntigenTestTypeId].ToString()))
            {
                if (!int.TryParse(testTypeMapping.Rows[0][VbecsTables.BloodTestTypeMapping.AntigenTestTypeId].ToString(), out mappedTypeId))
                {
                    validationErrors.Add(string.Format("PendingResultController.GetTestTypeMapping: Cannot convert MappedTypeId {0} to type int.",
                        testTypeMapping.Rows[0][VbecsTables.BloodTestTypeMapping.AntigenTestTypeId].ToString()));
                    return false;
                }
            }
            
            return true;
        }

        /// <Developers>
        /// 	<Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/28/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        ///  <summary>
        ///  Adds a row the updateTaskStatus datatable for the UpdateOrderedTestTaskStatus stored procedure.
        ///  </summary>
        /// <param name="updateTaskStatus"></param>
        /// <param name="pendingTest"></param>
        /// <param name="errorList"></param>
        /// <returns><see cref="bool"/>true if the row is added false if not.
        /// </returns>
        public static bool AddUpdateOrderedTestTaskStatus(ref DataTable updateTaskStatus, PendingTest pendingTest, ref List<string> errorList)
        {
            if (updateTaskStatus.Select("orderedtestguid=" + "'" + pendingTest.OrderedTestGuid + "'").Length > 0)
                return false;

            DataRow updateTaskStatusRow = updateTaskStatus.NewRow();

            if (pendingTest.OrderedTestGuid == Guid.Empty)
            {
                errorList.Add(
                    string.Format(
                        "PendingResultController.AddUpdateOrderedTestTaskStatus: OrderedTestGuid is null."));
                return false;
            }
            if (pendingTest.OrderedTestGuid == Guid.Empty)
                updateTaskStatusRow[VbecsTables.OrderedTest.OrderedTestGuid] = pendingTest.OrderedTestGuid;
            else
                updateTaskStatusRow[VbecsTables.OrderedTest.OrderedTestGuid] = pendingTest.OrderedTestGuid;

            updateTaskStatusRow[VbecsTables.OrderedTest.TaskStatusCode] = Utility.GetTaskStatusCodeFromEnum(TaskStatus.PendingReview);

            updateTaskStatusRow[VbecsTables.OrderedTest.RowVersion] = pendingTest.RowVersion;

            updateTaskStatusRow[VbecsTables.OrderedTest.LastUpdateFunctionId] = UpdateFunction.HL7AutoInstrumentInterface;

            updateTaskStatusRow[VbecsTables.OrderedTest.LastUpdateUser] = pendingTest.TestTechId;

            int rowCount = updateTaskStatus.Rows.Count;

            updateTaskStatus.Rows.Add(updateTaskStatusRow);
            if (updateTaskStatus.Rows.Count == rowCount + 1)
                return true;

            return false;
        }
        /// <Developers>
        /// 	<Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        ///  <summary>
        ///  Adds a row the the pendingResultTable.
        ///  </summary>
        /// <param name="pendingResultTable"></param>
        /// <param name="pendingTest"></param>
        /// <returns><see cref="bool"/>true if the row is added false if not.
        /// </returns>
        public static bool AddPendingSpecimenRow(ref DataTable pendingResultTable, PendingSpecimenTest pendingTest)
        {
            DataRow pendingResultRow = pendingResultTable.NewRow();

            PendingSpecimenTest pendingSpecimenTest = (PendingSpecimenTest)pendingTest;

            if (pendingSpecimenTest.OrderedTestGuid == Guid.Empty)
                pendingResultRow[VbecsTables.PendingSpecimenTest.OrderedTestGuid] = DBNull.Value;
            else
                pendingResultRow[VbecsTables.PendingSpecimenTest.OrderedTestGuid] = pendingSpecimenTest.OrderedTestGuid;

            if (pendingSpecimenTest.PatientSpecimenGuid == Guid.Empty)
                pendingResultRow[VbecsTables.PendingSpecimenTest.PatientSpecimenGuid] = DBNull.Value;
            else
                pendingResultRow[VbecsTables.PendingSpecimenTest.PatientSpecimenGuid] = pendingSpecimenTest.PatientSpecimenGuid;

            pendingResultRow[VbecsTables.PendingSpecimenTest.BloodTestTypeId] = pendingSpecimenTest.BloodTestTypeId;
            pendingResultRow[VbecsTables.PendingSpecimenTest.TestDate] = pendingSpecimenTest.TestDate;
            pendingResultRow[VbecsTables.PendingSpecimenTest.TestTechId] = pendingSpecimenTest.TestTechId;
            pendingResultRow[VbecsTables.PendingSpecimenTest.TestResultId] = pendingSpecimenTest.TestResultId;
            pendingResultRow[VbecsTables.PendingSpecimenTest.AutoInstrumentName] = pendingSpecimenTest.AutoInstrumentName;
            pendingResultRow[VbecsTables.PendingSpecimenTest.AutoInstrumentComments] = pendingSpecimenTest.AutoInstrumentComments;
            pendingResultRow[VbecsTables.PendingSpecimenTest.DivisionCode] = pendingSpecimenTest.DivisionCode;
            pendingResultRow[VbecsTables.PendingSpecimenTest.PendingTestStatusId] = (int)PendingTestStatus.Pending;
            pendingResultRow[VbecsTables.PendingSpecimenTest.SpecimenResultSetGuid] = pendingSpecimenTest.ResultSetGuid;
            pendingResultRow[VbecsTables.PendingSpecimenTest.LastUpdateUser] = pendingSpecimenTest.LastUpdateUser;
            pendingResultRow[VbecsTables.PendingSpecimenTest.LastUpdateFunctionId] = UpdateFunction.HL7AutoInstrumentInterface;

            int rowCount = pendingResultTable.Rows.Count;

            pendingResultTable.Rows.Add(pendingResultRow);
            if (pendingResultTable.Rows.Count == rowCount + 1)
                return true;

            return false;
        }

        /// <Developers>
        /// 	<Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        ///  <summary>
        ///  Adds a row to the pending results table.
        ///  </summary>
        /// <param name="pendingResultTable"></param>
        /// <param name="pendingTest"></param>
        /// <returns><see cref="bool"/>true if the row is added false if not.
        /// </returns>
        public static bool AddPendingUnitRow(ref DataTable pendingResultTable, PendingBloodUnitTest pendingTest)
        {
            DataRow pendingResultRow = pendingResultTable.NewRow();
 
            PendingBloodUnitTest pendingBloodUnit = pendingTest;
            if (pendingBloodUnit.BloodUnitTestGuid == Guid.Empty)
                pendingResultRow[VbecsTables.PendingBloodUnitTest.BloodUnitTestGuid] = DBNull.Value;
            else
                pendingResultRow[VbecsTables.PendingBloodUnitTest.BloodUnitTestGuid] = pendingBloodUnit.BloodUnitTestGuid;

            if (pendingBloodUnit.BloodUnitGuid == Guid.Empty)
                pendingResultRow[VbecsTables.PendingBloodUnitTest.BloodUnitGuid] = DBNull.Value;
            else
                pendingResultRow[VbecsTables.PendingBloodUnitTest.BloodUnitGuid] = pendingBloodUnit.BloodUnitGuid;

            if (pendingBloodUnit.OrderedUnitGuid == Guid.Empty)
                pendingResultRow[VbecsTables.PendingBloodUnitTest.OrderedUnitGuid] = DBNull.Value;
            else
                pendingResultRow[VbecsTables.PendingBloodUnitTest.OrderedUnitGuid] = pendingBloodUnit.OrderedUnitGuid;

            pendingResultRow[VbecsTables.PendingBloodUnitTest.BloodTestTypeId] = pendingBloodUnit.BloodTestTypeId;
            pendingResultRow[VbecsTables.PendingBloodUnitTest.BloodUnitId] = pendingBloodUnit.BloodUnitId;
            pendingResultRow[VbecsTables.PendingBloodUnitTest.TestDate] = pendingBloodUnit.TestDate;
            pendingResultRow[VbecsTables.PendingBloodUnitTest.TestTechId] = pendingBloodUnit.TestTechId;
            pendingResultRow[VbecsTables.PendingBloodUnitTest.TestResultId] = pendingBloodUnit.TestResultId;
            pendingResultRow[VbecsTables.PendingBloodUnitTest.AutoInstrumentComments] = pendingBloodUnit.AutoInstrumentComments;

            if (pendingBloodUnit.OrderedTestGuid == Guid.Empty)
                pendingResultRow[VbecsTables.PendingBloodUnitTest.OrderedTestGuid] = DBNull.Value;
            else
                pendingResultRow[VbecsTables.PendingBloodUnitTest.OrderedTestGuid] = pendingBloodUnit.OrderedTestGuid;

            if (pendingBloodUnit.OrderedComponentGuid == Guid.Empty)
                pendingResultRow[VbecsTables.PendingBloodUnitTest.OrderedComponentGuid] = DBNull.Value;
            else
                pendingResultRow[VbecsTables.PendingBloodUnitTest.OrderedComponentGuid] = pendingBloodUnit.OrderedComponentGuid;

            pendingResultRow[VbecsTables.PendingBloodUnitTest.AutoInstrumentName] = pendingBloodUnit.AutoInstrumentName;
            pendingResultRow[VbecsTables.PendingBloodUnitTest.DivisionCode] = pendingBloodUnit.DivisionCode;
            pendingResultRow[VbecsTables.PendingBloodUnitTest.PendingTestStatusId] = pendingBloodUnit.PendingTestStatusId;

            if (pendingBloodUnit.ResultSetGuid == Guid.Empty)
                pendingResultRow[VbecsTables.PendingBloodUnitTest.UnitResultSetGuid] = DBNull.Value;
            else
                pendingResultRow[VbecsTables.PendingBloodUnitTest.UnitResultSetGuid] = pendingBloodUnit.ResultSetGuid;

            pendingResultRow[VbecsTables.PendingBloodUnitTest.LastUpdateUser] = pendingBloodUnit.LastUpdateUser;
            pendingResultRow[VbecsTables.PendingBloodUnitTest.LastUpdateFunctionId] = UpdateFunction.HL7AutoInstrumentInterface;

            int rowCount = pendingResultTable.Rows.Count;

            pendingResultTable.Rows.Add(pendingResultRow);
            if (pendingResultTable.Rows.Count == rowCount + 1)
                return true;
            
            return false;
        }

        /// <Developers>
        /// <Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// Handles adding an interpritation for antigen tests.
        /// </summary>
        /// <param name="mappedTypeId"> Blood unit ID</param>
        /// <param name="pendingTest"> Blood unit ID</param>
        /// <param name="comment"></param>
        /// <param name="pendingDataTable"></param>
        /// <param name="updateOrderTable"></param>
        /// <param name="errorList"></param>
        /// <returns><see cref="bool"/>true if the specimen is found false if not.
        /// </returns>
        public static bool AntigenProcessing(int mappedTypeId, PendingTest pendingTest, string comment,
            ref DataTable pendingDataTable, ref DataTable updateOrderTable, ref List<string> errorList)
        {
            //Make a copy for the AntigenInterp insert.
            PendingTest antigenTestCopy;
            if (pendingTest is PendingBloodUnitTest)
            {
                antigenTestCopy = new PendingBloodUnitTest((PendingBloodUnitTest) pendingTest);
            }
            else
            {
                antigenTestCopy = new PendingSpecimenTest((PendingSpecimenTest) pendingTest);
            }

            //Add the mapped Id.
            antigenTestCopy.BloodTestTypeId = mappedTypeId;
            if (string.IsNullOrEmpty(comment))
                antigenTestCopy.AutoInstrumentComments = string.Empty;
            else
                antigenTestCopy.AutoInstrumentComments = comment;

            //Hash table lookup for testresult and antigentype
            //Result Type Text: 2+ ->mapped-> Result Type ID (VBECS): 2
            antigenTestCopy.TestResultId = AntigenLookup(antigenTestCopy, ref errorList);
            if (string.IsNullOrEmpty(antigenTestCopy.TestResultId))
            {
                errorList.Add(
                     string.Format("Not supported antigen test result ID: {0} on Test: {1} For Blood Test {2}.",
                         antigenTestCopy.TestResultId, antigenTestCopy.OrderableTestName, antigenTestCopy.BloodTestTypeName));
                return false; 
            }

            if (pendingTest is PendingBloodUnitTest)
            {
                AddPendingUnitRow(ref pendingDataTable, (PendingBloodUnitTest)antigenTestCopy);
            }
            else
            {
                AddPendingSpecimenRow(ref pendingDataTable, (PendingSpecimenTest)antigenTestCopy);
                //Ordered test task status should be changed to "Pending Review" since we received test results
                if (pendingTest.TaskStatusCode != Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.PendingReview).ToString())
                  AddUpdateOrderedTestTaskStatus(ref updateOrderTable, (PendingSpecimenTest)antigenTestCopy, ref errorList);                
            }

            return true;
        }

        /// <Developers>
        /// <Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// Split the ABORhInterp into the ABO pare and the Rh part.
        /// Add row for extra interp.
        /// ABORhInterp = "A Pos" = ABOInterp = "A", RhInterp = "N"
        /// </summary>
        /// <param name="testTypeId"> Blood unit ID</param>
        /// <param name="testComment"></param>
        /// <param name="pendingTest"> Blood unit ID</param>
        /// <param name="testResultText"></param>
        /// <param name="pendingDataTable"></param>
        /// <param name="updateOrderTable"></param>
        /// <param name="errorList"></param>
        /// <returns><see cref="bool"/>true if the specimen is found false if not.
        /// </returns>
        public static bool AboRhIterpProcessing(int testTypeId, string testComment, 
            PendingTest pendingTest, out string testResultText, ref DataTable pendingDataTable, ref DataTable updateOrderTable, 
             ref List<string> errorList)
        {
            testResultText = string.Empty;
            string[] interps = pendingTest.TestResultText.Split();

            if (interps.Count() != 2)
            {
                errorList.Add(
                     string.Format("Not supported ABORh intrepritaion invalid count: {0} on Test: {1} For Blood Test {2}.",
                         pendingTest.TestResultId, pendingTest.OrderableTestName, pendingTest.BloodTestTypeName));

                return false;
            }

            PendingTest aboInterpTestTypeCopy;
            //Make a copy for the ABOInterp insert.
            if(pendingTest is PendingBloodUnitTest)
                aboInterpTestTypeCopy = new PendingBloodUnitTest((PendingBloodUnitTest)pendingTest);
            else
                aboInterpTestTypeCopy = new PendingSpecimenTest((PendingSpecimenTest)pendingTest);

            aboInterpTestTypeCopy.TestResultText = interps[0];
            testResultText = interps[1];

            if (string.IsNullOrEmpty(testComment))
                aboInterpTestTypeCopy.AutoInstrumentComments = string.Empty;
            else
                aboInterpTestTypeCopy.AutoInstrumentComments = testComment;

            //Hash table lookup for testresult to the interp result type
            //A -> A
            aboInterpTestTypeCopy.TestResultId = InterpResultLookup(aboInterpTestTypeCopy, ref errorList);
            if (string.IsNullOrEmpty(aboInterpTestTypeCopy.TestResultId))
            {
                errorList.Add(
                     string.Format("Not supported ABORh intrepritaion result ID: {0} on Test: {1} For Blood Test {2}.",
                         aboInterpTestTypeCopy.TestResultId, aboInterpTestTypeCopy.OrderableTestName, aboInterpTestTypeCopy.BloodTestTypeName));
                return false;
            }

            //Add the ABOInterp row.
            if (pendingTest is PendingBloodUnitTest)
            {
                AddPendingUnitRow(ref pendingDataTable, (PendingBloodUnitTest)aboInterpTestTypeCopy);
            }
            else
            {
                AddPendingSpecimenRow(ref pendingDataTable, (PendingSpecimenTest)aboInterpTestTypeCopy);
                //Insert update task status row.
                if (pendingTest.TaskStatusCode != Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.PendingReview).ToString())
                    AddUpdateOrderedTestTaskStatus(ref updateOrderTable, (PendingSpecimenTest)aboInterpTestTypeCopy, ref errorList);
            }

            return true;
        }

        /// <Developers>
        /// <Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// Campares AntiD1 to AntiD2 if AntiD1 not equal AntiD2 then reject the message.
        /// If if AntiD1  equal AntiD2 assign AntiD to AntiD1 and remove AntiD2.
        /// </summary>
        /// <param name="pendingTests"></param>
        /// <param name="errorList"></param>
        /// <returns><see cref="string"/>true if the specimen is found false if not.
        /// </returns>
        public static void AntiD1AntiD2Compare(ref List<PendingTest> pendingTests, ref List<string> errorList)
        {
            var antiDList = pendingTests.FindAll(x => x.BloodTestTypeName.Contains("AntiD"));
            if (antiDList.Any() && antiDList.Count == 2)
            {
                PendingSpecimenTest antiD1 = (PendingSpecimenTest) antiDList[0];
                PendingSpecimenTest antiD2 = (PendingSpecimenTest) antiDList[1];

                //Hash table lookup for testresult and antigentype
                if (!TestResultLookup.Contains(antiD1.TestResultText))
                {
                    errorList.Add(string.Format("Not supported AntiD1 test result ID: {0} on Test: {1} For Blood Test {2}.",
                            antiD1.TestResultId, antiD1.OrderableTestName, antiD1.BloodTestTypeName));
                    return;
                }

                //Hash table lookup for testresult and antigentype
                if (!TestResultLookup.Contains(antiD2.TestResultText))
                {
                    errorList.Add(
                        string.Format("Not supported AntiD2 test result ID: {0} on Test: {1} For Blood Test {2}.",
                            antiD2.TestResultId, antiD2.OrderableTestName, antiD2.BloodTestTypeName));
                    return;
                }

                if (AntigenLookUp[antiD1.TestResultText] != AntigenLookUp[antiD2.TestResultText])
                {
                    errorList.Add(
                        string.Format(
                            "Not supported AntiD1 test result {0} is not equal to AntiD2 test result ID: {1} on Test: {2} For Blood Test {3}.",
                            antiD1.TestResultText, antiD2.TestResultText, antiD2.OrderableTestName, antiD2.BloodTestTypeName));
                    return;
                }
                pendingTests.RemoveAll(x => x.BloodTestTypeName.Contains("AntiD1"));
                pendingTests.RemoveAll(x => x.BloodTestTypeName.Contains("AntiD2"));
                antiD1.BloodTestTypeName = "AntiD";
                pendingTests.Add(antiD1);
            }
        }

        /// <Developers>
        /// <Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// Handle  Crossmatch Interpritations where they are ISInterp and AHGInterp 
        /// IF ISIntper = C and AHGInterp = I the combined iterp = CI this is mapped to I.
        /// If their is only one Interp process it normally.
        /// If theree is a missing IS test it replaces it with IS row with resultId of X.
        /// </summary>
        /// <param name="pendingTests"></param>
        /// <param name="errorList"></param>
        /// <returns><see cref="string"/>true if the specimen is found false if not.
        /// </returns>
        public static void XMInterpISHandling(ref List<PendingTest> pendingTests, ref List<string> errorList)
        {

            var xmInterpList = pendingTests.FindAll(x => x.BloodTestTypeName.Contains("Interp"));
            if (xmInterpList.Any() && xmInterpList.Count == 2)
            {
                PendingBloodUnitTest interp1 = (PendingBloodUnitTest)xmInterpList[0];
                PendingBloodUnitTest interp2 = (PendingBloodUnitTest)xmInterpList[1];

                //Hash table lookup for Interpritation.
                if (!XMInterpLookup.Contains(interp1.TestResultText))
                {
                    errorList.Add(
                        string.Format("Not supported Interp: {0} on Test: {1} For Blood Test {2}.",
                            interp1.TestResultText, interp1.OrderableTestName, interp1.BloodTestTypeName));
                    return;
                }

                //Hash table lookup for Interpritation
                if (!XMInterpLookup.Contains(interp2.TestResultText))
                {
                    errorList.Add(
                        string.Format("Not supported Interp: {0} on Test: {1} For Blood Test {2}.",
                            interp2.TestResultText, interp2.OrderableTestName, interp2.BloodTestTypeName));
                    return;
                }

                //Create a combined Interpretation eg. C + C = CC.
                string combinedInterp = (string)XMInterpLookup[interp1.TestResultText] + (string)XMInterpLookup[interp2.TestResultText];


                //Lookup a combined Interpretation eg. CC = C.
                if (!XMInterpCombinedLookup.Contains(combinedInterp))
                {
                    errorList.Add(
                        string.Format("Not supported Interp: {0} on Test: {1} For Blood Test {2}.",
                            interp1.TestResultText, interp1.OrderableTestName, interp1.BloodTestTypeName));
                    return;
                }

                //Construct a copy of the first interpritaion.
                PendingBloodUnitTest combinedInterpPendingTest = new PendingBloodUnitTest(interp1);

                //Make it into an XMInterp interpretation and adding to the set of tests.
                combinedInterpPendingTest.BloodTestTypeName = "XMInterp";
                combinedInterpPendingTest.TestResultText = (string)XMInterpCombinedLookup[combinedInterp];
                combinedInterpPendingTest.TestResultId = (string)XMInterpCombinedLookup[combinedInterp];
                pendingTests.Add(combinedInterpPendingTest);
                //TOD: Say why Zero
                //Add the interpretations for and set the OrderableTEstId to 0 for later processing.
                interp1.TestResultId = (string)XMInterpLookup[interp1.TestResultText];
                interp1.OrderableTestId = 0;
                interp2.TestResultId = (string)XMInterpLookup[interp2.TestResultText];
                interp2.OrderableTestId = 0;
            }
            else if (xmInterpList.Any() && xmInterpList.Count == 1)
            {
                PendingTest interp1 = xmInterpList[0];

                //Hash table lookup for Interpretation.
                if (!XMInterpLookup.Contains(interp1.TestResultText))
                {
                    errorList.Add(
                        string.Format("Not supported Interp: {0} on Test: {1} For Blood Test {2}.",
                            interp1.TestResultText, interp1.OrderableTestName, interp1.BloodTestTypeName));
                    return;
                }
                interp1.TestResultId = (string)XMInterpLookup[interp1.TestResultText];
            }

            //Determine if there is an IS test for this crossmatch test.  
            //If therer isn't one then add and IS test with result of X.
            if (pendingTests.Any() && !pendingTests.Exists(x => x.BloodTestTypeName == "IS"))
            {
                PendingBloodUnitTest isRow = new PendingBloodUnitTest((PendingBloodUnitTest)pendingTests[0]);
                isRow.BloodTestTypeName = "IS";
                isRow.TestResultText = "X";
                isRow.TestResultId = "X";
                pendingTests.Add(isRow);
            }
        }

        /// <Developers>
        /// <Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// Look verify and assign the test result.  1 -> 1 
        /// </summary>
        /// <param name="pendingTest"></param>
        /// <param name="errorList"></param>
        /// <returns><see cref="string"/>true if the specimen is found false if not.
        /// </returns>
        public static string ResultLookup( PendingTest pendingTest, ref List<string> errorList)
        {
            //Hash table lookup for testresult and antigentype
            if (!TestResultLookup.Contains(pendingTest.TestResultText))
            {
                errorList.Add(string.Format("Not supported test result ID: {0} on Test: {1} For Blood Test {2}.",
                      pendingTest.TestResultId, pendingTest.OrderableTestName, pendingTest.BloodTestTypeName));
                return null;
            }
            return TestResultLookup[pendingTest.TestResultText].ToString();
        }

        /// <Developers>
        /// <Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// Look verify and assign the antigen test Interpretation.  1 -> p 
        /// </summary>
        /// <param name="pendingTest"></param>
        /// <param name="errorList"></param>
        /// <returns><see cref="string"/>true if the specimen is found false if not.
        /// </returns>
        public static string AntigenLookup( PendingTest pendingTest, ref List<string> errorList)
        {
            //Hash table lookup for testresult and antigentype
            if (!AntigenLookUp.Contains(pendingTest.TestResultText))
            {
                errorList.Add(string.Format("Not supported test result ID: {0} on Test: {1} For Blood Test {2}.",
                    pendingTest.TestResultText, pendingTest.OrderableTestName, pendingTest.BloodTestTypeName));
                return null;
            }
            return AntigenLookUp[pendingTest.TestResultText].ToString();
        }

        /// <Developers>
        /// <Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// Look verify and assign the Crossmatch test Interpretation.  Comp -> C 
        /// </summary>
        /// <param name="pendingTest"></param>
        /// <param name="errorList"></param>
        /// <returns><see cref="string"/>true if the specimen is found false if not.
        /// </returns>
        public static string XMInterpResultLookup(PendingTest pendingTest, ref List<string> errorList)
        {
            //Hash table lookup for testresult and antigentype
            if (!XMInterpLookup.Contains(pendingTest.TestResultText))
            {
                errorList.Add(string.Format("Not supported test interpritation ID: {0} on Test: {1} For Blood Test {2}.",
                    pendingTest.TestResultText, pendingTest.OrderableTestName, pendingTest.BloodTestTypeName));
                return null;
            }
            return XMInterpLookup[pendingTest.TestResultText].ToString();
        }


        /// <Developers>
        /// <Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// Look verify and assign the  test Interpretation.  Pos -> P, A -> A 
        /// </summary>
        /// <param name="pendingTest"></param>
        /// <param name="errorList"></param>
        /// <returns><see cref="string"/>true if the specimen is found false if not.
        /// </returns>
        public static string InterpResultLookup(PendingTest pendingTest, ref List<string> errorList)
        {
            //Hash table lookup for testresult and antigentype
            if (!InterpLookup.Contains(pendingTest.TestResultText))
            {
                errorList.Add(string.Format("Not supported test interpritation ID: {0} on Test: {1} For Blood Test {2}.",
                    pendingTest.TestResultText, pendingTest.OrderableTestName, pendingTest.BloodTestTypeName));
                return null;
            }
            return InterpLookup[pendingTest.TestResultText].ToString();
        }

        /// <Developers>
        /// <Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// BR_114.04	
        /// A received order currently pending review must be rejected by the user in VBECS before replacement results can be received.            
        /// Check if there are pending blood tests with the same UnitId and pending status. If there are throw an exception. 
        /// </summary>
        /// <param name="specimenUID"> Blood unit ID</param>
        /// <param name="divisionCode"></param>
        /// <param name="bloodTestType"></param>
        /// <returns><see cref="bool"/>true if the specimen is found false if not.
        /// </returns>
        public static bool PendingSpecimenTestExists(string specimenUID, string divisionCode, int bloodTestType)
        {
            // BR_114.04	
            //A received order currently pending review must be rejected by the user in VBECS before replacement results can be received.           
            // Check if there are pending blood tests with the same UnitId and pending status. If there are throw an exception. 
            DataTable pendingSpecimenTestTable;
            pendingSpecimenTestTable = GetPendingSpecimenTest(specimenUID, divisionCode, PendingTestStatus.Pending, bloodTestType);
            return pendingSpecimenTestTable != null && pendingSpecimenTestTable.Rows.Count > 0;
        }


        /// <Developers>
        /// <Developer>Russell Stephenson</Developer>
        /// </Developers>
        /// <SiteName>Hines OIFO</SiteName>
        /// <CreationDate>1/16/2016</CreationDate>
        /// <TestCases>
        /// </TestCases>
        /// <Update></Update>
        /// <ArchivePlan></ArchivePlan>
        /// <Interfaces></Interfaces>
        /// <summary>
        /// BR_114.04	
        /// A received order currently pending review must be rejected by the user in VBECS before replacement results can be received.            
        /// Check if there are pending blood tests with the same UnitId and pending status. If there are throw an exception. 
        /// </summary>
        /// <param name="unitId"> Blood unit ID</param>
        /// <param name="divisionCode"></param>
        /// <param name="bloodTestType"></param>
        /// <returns><see cref="bool"/>true if the blood unit is found false if not.
        /// </returns>
        public static bool PendingBloodUnitTestExists(string unitId, string divisionCode, int bloodTestType)
        {
            // BR_114.04	
            //A received order currently pending review must be rejected by the user in VBECS before replacement results can be received.           
            // Check if there are pending blood tests with the same UnitId and pending status. If there are throw an exception. 
            DataTable pendingBloodUnitTestTable = GetPendingBloodUnitTest(unitId, divisionCode, PendingTestStatus.Pending, bloodTestType);
            return pendingBloodUnitTestTable!= null && pendingBloodUnitTestTable.Rows.Count > 0;
        }

        ///  <Developers>
        ///  <Developer>Russell Stephenson</Developer>
        ///  </Developers>
        ///  <SiteName>Hines OIFO</SiteName>
        ///  <CreationDate>1/16/2016</CreationDate>
        ///  <TestCases>
        ///  </TestCases>
        ///  <Update></Update>
        ///  <ArchivePlan></ArchivePlan>
        ///  <Interfaces></Interfaces>
        ///  <summary>
        ///  BR_114.09	The system stores the following information:
        /// 	Test Results
        /// 	Test Interpretations
        /// 	Testing Tech ID
        ///     Automated Instrument identifier
        ///     Date and Time Tested
        ///     Automated Instrument Test Comments  
        ///  </summary>
        ///  <param name="message"> Blood unit ID</param>
        /// <param name="bloodTestPanelName"></param>
        public static List<string> ProcessTestResults(HL7AIOruMessage message, BloodTestPanelName bloodTestPanelName)
        {            
            bool isSpecimen = false;
            List<string> validationErrors = new List<string>();

            // Facilitates BR_114.01 
            // If the blood unit record is not found, the message is rejected.           
            DataTable bloodUnitTable = GetBloodUnit(message.Obr.UnitId, message.Msh.ReceivingFacility);
            bool isBloodUnit = bloodUnitTable != null &&  bloodUnitTable.Rows.Count > 0;

            //Defect 303325 test for duplicate test types.
            if (message.ObxSegments.GroupBy(s => s.TestTypeId).SelectMany(grp => grp.Skip(1)).Any())
            {
                validationErrors.Add(string.Format("Duplcate Tests in Message OBX.3."));
            }

            bool isTestResultStatus = message.ObxSegments.Where(x => HL7Utility.GetObservationResultStatusEnumFromChar(x.SpecimenActionCode) != ObservationResultStatus.Final).Any();
            if (isTestResultStatus)
                validationErrors.Add(string.Format("Invalid Observation Result Status OBX.11."));

            ArrayList sprocArray = new ArrayList();
            ArrayList dtArray = new ArrayList();


            if (bloodTestPanelName == BloodTestPanelName.XM)
            {
                //BR_114.03	
                //If the cross-matched units are not assigned to a patient order, the message containing crossmatch results is rejected.
                //If we found any blood units for the unitID we need to check if this is a crossmatch.
                if (!isBloodUnit)
                {
                 validationErrors.Add(string.Format("No Crossmatch Blood Unit  Available for Identifier: {0}.",
                    message.Obr.UnitId));
                    return validationErrors;
                }
                //check if we have at least one Crossmatch if we have match for both the specimenUID and the bloodUnitId we have a crossmatch.
                //So we set the isSpecimen flag to true as well as the isBloodUnit flag so we know it is a crossmatch.
                bloodUnitTable = HL7GetXMBySpecimenUIDAndUnitID(message.Obr.SpecimenUid, message.Obr.UnitId,
                    message.Msh.ReceivingFacility);
                isSpecimen = bloodUnitTable.Rows.Count > 0;

                if (!isSpecimen)
                {
                   
                    validationErrors.Add(string.Format("No Crossmatch Specimen Available for Identifier: {0}.",
                    message.Obr.SpecimenUid));
                    return validationErrors;
                }
            }

            DataTable specimenTable = null;
            //If we dont have a crossmatch we test for specimen test. 
            //If isSpecimen is false then we don't have a crossmatch.
            //Check if we have at least one specimen
            if (!isSpecimen)
            {
                //BR_114.02
                //If the patient specimen associated with an active patient order is not found, the message is rejected.
                specimenTable = GetSpecimen(message.Obr.SpecimenUid, message.Obr.TestName, string.Empty, message.Msh.ReceivingFacility);
                isSpecimen = specimenTable != null && specimenTable.Rows.Count > 0;
            }

            if (isBloodUnit && isSpecimen) //true&true So we have a crossmatch.
            {
                string comment = string.Empty;

                //Initialize the pending Crossmatch data for processing.  This is done on the blood unit path.
                IList<PendingTest> pendingTests = GetPendingBloodUnitTests(message, ref bloodUnitTable, out comment,
                    bloodTestPanelName, ref validationErrors);

                //Crossmatch is handled by the Blood Unit processing path. All paths are processed with this method.
                validationErrors.AddRange(ProcessPendingTestResults((List<PendingTest>)pendingTests, 
                    comment, ref sprocArray, ref dtArray, bloodTestPanelName));
                if (validationErrors.Count == 0)
                {
                    try
                    {
                        new StoredProcedure().TransactionalGetValue(sprocArray, dtArray);
                    }
                    catch (Exception ex)
                    {
                        throw new DataAccessLayerException("Internal error: Crossmatch Insert Failed.", ex);
                    }
                }
                return validationErrors;
            }
            if (!isSpecimen && isBloodUnit) //false&true  We have a blood unit.
            {
                string comment = string.Empty;

                //Initialize the pending Blood Unit data for processing.  This is done on the blood unit path.
                IList<PendingTest> pendingTests = GetPendingBloodUnitTests(message, ref bloodUnitTable, out comment,
                    bloodTestPanelName, ref validationErrors);

                //Process the blood unit test data.  All paths are processed with this method.
                validationErrors.AddRange(ProcessPendingTestResults((List<PendingTest>)pendingTests,
                    comment, ref sprocArray, ref dtArray, bloodTestPanelName));
                if (validationErrors.Count == 0)
                {
                    try
                    {
                        new StoredProcedure().TransactionalGetValue(sprocArray, dtArray);
                    }
                    catch (Exception ex)
                    {
                        throw new DataAccessLayerException("Internal error: Blood Unit Insert Failed.", ex);
                    }
                }
                return validationErrors;
            }
            if (isSpecimen)     //true&false  We have a specimen and not a blood unit.
            {
                if (specimenTable == null)
                {
                    validationErrors.Add("No specimen data for specimen results found");
                    return validationErrors;
                }

                //BR_114.07	
                //If test results currently exist for a patient order (entered manually or by the Automated Instrument), 
                //the replacement test results will not be accepted into VBECS until the user invalidates the existing test results.
                if (specimenTable.Rows.Count > 1 && (bloodTestPanelName != BloodTestPanelName.AnigenTyping 
                    && bloodTestPanelName != BloodTestPanelName.DAT))
                {
                    //BR_114.10	
                    //When test results for tests listed in PT_114.01 are sent to VBECS and multiple orders for the same test type with the same 
                    //specimen UID on the VBECS PTL, the message is rejected.
                    //For example: When an ABS and a Repeat ABS are on the PTL with the same specimen UID, results cannot be aligned from the Automated Instrument.
                    string testName = specimenTable.Rows[0][VbecsTables.OrderableTest.OrderableTestName] == DBNull.Value
                        ? string.Empty
                        : specimenTable.Rows[0][VbecsTables.OrderableTest.OrderableTestName].ToString();
                    validationErrors.Add(
                        string.Format("Multiple Tests Exist for Specimen UID: {0} and Test Name: {1}.",
                            message.Obr.SpecimenUid, testName));
                    return validationErrors;
                }

                string comment;
                //Initialize the pending Specimen Test data for processing.  This is done on the specimen test path.
                IList<PendingTest> pendingTests = GetPendingSpecimenTests(message, out comment, bloodTestPanelName, ref validationErrors);
                if (validationErrors.Count > 0)
                    return validationErrors;

                //Process the Specimen test data.  All paths are processed with this method.
                validationErrors.AddRange(ProcessPendingTestResults((List<PendingTest>)pendingTests, 
                    comment, ref sprocArray, ref dtArray, bloodTestPanelName));
                if (validationErrors.Count == 0)
                {
                    try
                    {
                        new StoredProcedure().TransactionalGetValue(sprocArray, dtArray);
                    }
                    catch (Exception ex)
                    {
                        throw new DataAccessLayerException("Internal error: Patient Specimen Insert Failed.", ex);
                    }
                }

                return validationErrors;
            } //false&false

            //Check for multiple blood units if there are multiple blood units then don't put the BloodUnitGuid in the table.
            if (message.Obr.UnitId == message.Obr.SpecimenUid)
            {
                validationErrors.Add(string.Format("No Blood Unit Or Specimen Available for Identifier: {0}.",
                    message.Obr.UnitId));
            }
            else
            {
                validationErrors.Add(string.Format("No Blood Unit Or Specimen Available for Specimen UID: {0} and Blood Unit ID: {1}.",
                    message.Obr.SpecimenUid, message.Obr.UnitId));
            }

            return validationErrors;
     
        }

        /// <summary>
        /// This method intializes the Blood Unit and Crossmatch data.
        /// Data is store in a List of PeningBloodUnit objects.
        /// </summary>
        /// <param name="message"></param>
        /// <param name="bloodUnitTable"></param>
        /// <param name="comment"></param>
        /// <param name="bloodTestPanelName"></param>
        /// <param name="validationErrors"></param>
        /// <returns></returns>
        public static IList<PendingTest> GetPendingBloodUnitTests(HL7AIOruMessage message, ref DataTable bloodUnitTable,
            out string comment, BloodTestPanelName bloodTestPanelName, ref List<string> validationErrors)
        {


            //Begin initializing common test data.
            Guid resultSetGuid = Guid.NewGuid();

            //Assign the auto istrument comment that is contained in the message.
            comment = message.Obr.TestComment;

            //Assign the blood unit guid for adding to all the rows of the pendingTest data.
            Guid bloodUnitGuid = bloodUnitTable.Rows.Count > 1
                ? Guid.Empty
                : (Guid)bloodUnitTable.Rows[0][VbecsTables.BloodUnit.BloodUnitGuid];

            int orderableTestId = 0;
            Guid orderedTestGuid = Guid.Empty;
            Guid orderedUnitGuid = Guid.Empty;
            Guid orderedComponentGuid = Guid.Empty;
            Guid patientSpecimenGuid = Guid.Empty;
            Guid bloodUnitStatusGuid = Guid.Empty;
            Byte[] rowVersion = {0};
            if (bloodTestPanelName == BloodTestPanelName.XM)
            {
                //Here we test if we have an Ordered Test Guid or and Ordered Component Guid.
                //The existance of an Ordered Test Guid means that this is a repeat crossmatch test.
                //The existance of an Ordered Component Grid means that this is is a regular crossmatch test.          
                if (bloodUnitTable.Rows.Count > 1)
                {
                    bool hasOrderedTestGuid =
                        bloodUnitTable.Select("orderedtestguid<>" + "'" + DBNull.Value + "'").Length > 0;
                    bool hasOrderedComponentGuid =
                        bloodUnitTable.Select("orderedcomponentguid<>" + "'" + DBNull.Value + "'").Length > 0;

                    if (hasOrderedTestGuid && hasOrderedComponentGuid)
                        validationErrors.Add(
                            string.Format("Test and Repeat Test Found For BloodUnitId: {0} and SpecimenUid: {1}.",
                                message.Obr.UnitId, message.Obr.SpecimenUid));
                }

                orderableTestId = bloodUnitTable.Rows[0][VbecsTables.OrderableTest.OrderableTestId] == DBNull.Value
                    ? (int) Common.OrderableTest.XM
                    : (int) bloodUnitTable.Rows[0][VbecsTables.OrderableTest.OrderableTestId];
                

                 orderedTestGuid = bloodUnitTable.Rows[0][VbecsTables.PendingBloodUnitTest.OrderedTestGuid] == DBNull.Value
                    ? Guid.Empty
                    : (Guid)bloodUnitTable.Rows[0][VbecsTables.PendingBloodUnitTest.OrderedTestGuid];

                //Check for multiple bloodunits. There there are more than one then set orderedUnitGuid to empty.
                 if (bloodUnitTable.Rows.Count == 1)
                 {
                     orderedUnitGuid = bloodUnitTable.Rows[0][VbecsTables.PendingBloodUnitTest.OrderedUnitGuid] ==
                                       DBNull.Value
                         ? Guid.Empty
                         : (Guid)bloodUnitTable.Rows[0][VbecsTables.PendingBloodUnitTest.OrderedUnitGuid];
                 }
                 else
                 {
                     orderedUnitGuid = Guid.Empty;
                 }

                 orderedComponentGuid = bloodUnitTable.Rows[0][VbecsTables.PendingBloodUnitTest.OrderedComponentGuid] == DBNull.Value
                    ? Guid.Empty
                    : (Guid)bloodUnitTable.Rows[0][VbecsTables.PendingBloodUnitTest.OrderedComponentGuid];

                 patientSpecimenGuid = bloodUnitTable.Rows[0][VbecsTables.PendingSpecimenTest.PatientSpecimenGuid] == DBNull.Value
                    ? Guid.Empty
                    : (Guid)bloodUnitTable.Rows[0][VbecsTables.PendingSpecimenTest.PatientSpecimenGuid];

                 rowVersion = bloodUnitTable.Rows[0][VbecsTables.OrderedTest.RowVersion] == DBNull.Value
                     ? null
                     : (Byte[])bloodUnitTable.Rows[0][VbecsTables.OrderedTest.RowVersion];

            }


            //Loop through all of the OBX segments and assign processing data.
            IList<PendingTest> pendingTests = new List<PendingTest>();
            foreach (var obx in message.ObxSegments)
            {
                PendingBloodUnitTest currentPendingTest = new PendingBloodUnitTest();

                //Skip the comments.
                if (obx.TestTypeId.Contains("Comment"))
                {
                    continue;
                }

                //These are added by the call to get XM data.
                currentPendingTest.BloodUnitGuid = bloodUnitGuid;
                currentPendingTest.OrderedTestGuid = orderedTestGuid;
                currentPendingTest.OrderedUnitGuid = orderedUnitGuid;
                currentPendingTest.OrderedComponentGuid = orderedComponentGuid;
                currentPendingTest.PatientSpecimenGuid = patientSpecimenGuid;
                currentPendingTest.OrderableTestId = orderableTestId;

                //Remains
                currentPendingTest.BloodUnitStatusGuid = bloodUnitStatusGuid;

                //These come from the message.
                currentPendingTest.AutoInstrumentComments = comment;
                currentPendingTest.OrderableTestName = message.Obr.TestName;
                currentPendingTest.TestTechId = obx.TestTechId;
                currentPendingTest.BloodTestTypeName = obx.TestTypeId;
                currentPendingTest.TestDate = obx.TestDateTime;
                currentPendingTest.TestResultText = obx.TestResultId;
                currentPendingTest.DivisionCode = message.Msh.ReceivingFacility;
                currentPendingTest.PendingTestStatusId = (int)PendingTestStatus.Pending;
                currentPendingTest.AutoInstrumentName = message.Obr.InstrumentName;
                currentPendingTest.RowVersion = rowVersion;
                currentPendingTest.BloodUnitId = message.Obr.UnitId;
                currentPendingTest.ResultSetGuid = resultSetGuid;
                pendingTests.Add(currentPendingTest);
            }

            return pendingTests;
        }

        /// <summary>
        /// This method intializes the Specimen Test data.
        /// Data is store in a List of PeningSpecimenTest objects.
        /// </summary>
        /// <param name="message"></param>
        /// <param name="comment"></param>
        /// <param name="bloodTestPanelName"></param>
        /// <param name="validationErrors"></param>
        /// <returns></returns>
        public static IList<PendingTest> GetPendingSpecimenTests(HL7AIOruMessage message, out string comment, BloodTestPanelName bloodTestPanelName,
            ref List<string> validationErrors)
        {
            IList<PendingTest> pendingTests = new List<PendingTest>();

            comment = message.Obr.TestComment;

            if (message.ObxSegments == null)
                return pendingTests;

            PendingSpecimenTest currentPendingTest;
            //assuming it is specimen
            //We loop through each OBX to get the OrderedTestGuid, PatientSpecimenGuid and the OrderableTestId for each row.
            foreach (var obx in message.ObxSegments)
            {
                currentPendingTest = new PendingSpecimenTest();


                //Skip the comments.
                if (obx.TestTypeId.Contains("Comment"))
                    continue;

                currentPendingTest.OrderableTestName = message.Obr.TestName;
                currentPendingTest.TestTechId = obx.TestTechId;
                currentPendingTest.BloodTestTypeName = obx.TestTypeId;
                currentPendingTest.TestDate  = obx.TestDateTime;
                currentPendingTest.TestResultText = obx.TestResultId;
                currentPendingTest.DivisionCode = message.Msh.ReceivingFacility;
                currentPendingTest.PendingTestStatusId = (int)PendingTestStatus.Pending;
                currentPendingTest.AutoInstrumentName = message.Obr.InstrumentName;
                currentPendingTest.SpecimenUid = message.Obr.SpecimenUid;
                //Check the order for the specimen.
                DataTable specimenTable = GetSpecimen(message.Obr.SpecimenUid, message.Obr.TestName, obx.TestTypeId, message.Msh.ReceivingFacility);

                //If there is an order for the Specimen
                if (specimenTable != null && specimenTable.Rows.Count == 1 && specimenTable.Rows[0].ItemArray.Any())
                {
                    currentPendingTest.OrderedTestGuid = (Guid)specimenTable.Rows[0][VbecsTables.OrderedTest.OrderedTestGuid];
                    currentPendingTest.ResultSetGuid = currentPendingTest.OrderedTestGuid;
                    currentPendingTest.PatientSpecimenGuid = (Guid)specimenTable.Rows[0][VbecsTables.OrderedTest.PatientSpecimenGuid];
                    currentPendingTest.RowVersion = (Byte[])specimenTable.Rows[0][VbecsTables.OrderedTest.RowVersion];
                    currentPendingTest.OrderName = specimenTable.Rows[0][VbecsTables.OrderableTest.OrderableTestName].ToString();
                    int orderableTestId = 0;
                    if (!int.TryParse(specimenTable.Rows[0][VbecsTables.OrderableTest.OrderableTestId].ToString(), out orderableTestId))
                    {
                        validationErrors.Add(
                            string.Format(
                                "PendingResultController.GetTestTypeMapping: Cannot convert TestTypeId: {0} to type int for test name: {1}.",
                                specimenTable.Rows[0][VbecsTables.OrderableTest.OrderableTestId], currentPendingTest.OrderName));
                    }
                    else
                    {
                        currentPendingTest.OrderableTestId = orderableTestId;
                    }
                    currentPendingTest.TaskStatusCode = specimenTable.Rows[0][VbecsTables.OrderedTest.TaskStatusCode].ToString();
                    pendingTests.Add(currentPendingTest);
                    continue;
                }
                
                if (specimenTable.Rows.Count > 1)
                {
                    //BR_114.10	
                    //When test results for tests listed in PT_114.01 are sent to VBECS and multiple orders for the same test type with the same 
                    //specimen UID on the VBECS PTL, the message is rejected.
                    //For example: When an ABS and a Repeat ABS are on the PTL with the same specimen UID, results cannot be aligned from the Automated Instrument.
                    string testName = specimenTable.Rows[0][VbecsTables.OrderableTest.OrderableTestName] == null
                        ? string.Empty
                        : specimenTable.Rows[0][VbecsTables.OrderableTest.OrderableTestName].ToString();
                    validationErrors.Add(
                        string.Format("Multiple Tests Exist for Specimen UID: {0} and Test Name: {1}.",
                            message.Obr.SpecimenUid, testName));
                    return pendingTests;
                }
                else if (specimenTable.Rows.Count < 1)
                {
                    int testTypeId;
                    int mappedTypeId;
                    //PT_114.01	HL7 Tests Accepted by VBECS
                    //GetTestTypeMapping returns 0 for mappedTypeId if the table contains null.
                    //   Test Type Name (HL7)     testTypeId (Test Type Name (VBECS))       mappedTypeId (Mapped to Interpretation)
                    //        AntiE ->                     34 (Anti E) ->                              64 (AntiEInterp)
                    //pendingTest.OrderableTestName="Antigen Typing" (OBR)
                    //pendingTest.OrderableTestId=nnn (Looked up in VBECS, exist only for Specimen)
                    //pendingTest.BloodTestTypeName=AntiE (OBX)
                    GetTestTypeMapping(currentPendingTest.OrderableTestName, currentPendingTest.BloodTestTypeName,
                        currentPendingTest.OrderableTestId, out testTypeId, out mappedTypeId, ref validationErrors);

                    //BR_114.02
                    //If the patient specimen associated with an active patient order is not found, the message is rejected.
                    // Facilitates BR_114.01 
                    // If the blood unit record is not found, the message is rejected.
                    //Check for multiple blood units if there are multiple blood units then don't put the BloodUnitGuid in the table.
                    validationErrors.Add(
                        string.Format("No Blood Unit Or Specimen Available for Identifier: {0} and Test Name: {1}.",
                            message.Obr.SpecimenUid, obx.TestTypeId));
                    return pendingTests;
                }

            }
            return pendingTests;
        }

        /// <summary>
        ///  BR_114.09	The system stores the following information:
        /// 	Test Results
        /// 	Test Interpretations
        /// 	Testing Tech ID
        ///     Automated Instrument identifier
        ///     Date and Time Tested
        ///     Automated Instrument Test Comments  
        /// </summary>
        /// <param name="pendingTests"></param>
        /// <param name="comment"></param>
        /// <param name="sprocArray"></param>
        /// <param name="dtArray"></param>
        /// <param name="bloodTestPanelName"></param>
        /// <returns></returns>
        public static List<string> ProcessPendingTestResults(List<PendingTest> pendingTests,
            string comment, ref ArrayList sprocArray, ref ArrayList dtArray,
            BloodTestPanelName bloodTestPanelName)
        {

            //Use this flag to select antigen typing logic.
            // bool isAntigenType = false;OK than
            // if (bloodTestPanelName == BloodTestPanelName.AnigenTyping)
            //     isAntigenType = true;
            List<string> validationErrors = new List<string>();

            //Use this flag to select antigen typing logic.
            bool isAntigenType = false;
            if (bloodTestPanelName == BloodTestPanelName.AnigenTyping)
                isAntigenType = true;

            //Gets the first pending test and assigns it to the type.
            var pendingType = pendingTests.FirstOrDefault();
            bool isBloodUnitType = pendingType is PendingBloodUnitTest;

            //Construct the data table to put the pending results in.
            DataTable pendingSpecimenTestsResultTable = GetInsertPendingPatientSpecimenTestSprocTable();
            DataTable pendingBloodUnitResultTable = GetInsertPendingBloodUnitTestSprocTable();
            DataTable updateOrderTaskStatusTable = GetOrderTaskStatusSprocTable();

            if (bloodTestPanelName == BloodTestPanelName.PatientABORh || bloodTestPanelName == BloodTestPanelName.TAS)
            {
                //BR_114.08	
                //When the patient ABO/Rh message contains results for Anti-D1 and Anti-D2 which do not conflict (both are positive or both are negative), 
                //Anti-D1 only will be stored in VBECS.
                //BR_114.11	
                //When the ABO/Rh message contains conflicting results for Anti-D1 and Anti-D2 (one is positive and the other negative), the message will be rejected.
                //We need to compare the results for AntiD1 and AntiD2 if they are both neg and both pos we store the AntiD1 in AntiD and dtop AntiD2.
                //If AntiD1 is not equal to AntiD2 we reject the message. 
                AntiD1AntiD2Compare(ref pendingTests, ref validationErrors);
            }
            else if (bloodTestPanelName == BloodTestPanelName.XM)
            {
                XMInterpISHandling(ref pendingTests, ref validationErrors);

                if(pendingTests[0].OrderedTestGuid != Guid.Empty && pendingTests[0].TaskStatusCode != Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.PendingReview).ToString())
                    AddUpdateOrderedTestTaskStatus(ref updateOrderTaskStatusTable,  pendingTests[0], ref validationErrors);                                            
            }

            var lookedupUsers = new Dictionary<string, DataRow>();  //251417, changing <string, string> to <string, DataRow> to nab email later
            string userEmailForLockingFailure = null;

            #region Pending Results Loop

            foreach (var pendingTest in pendingTests)
            {
                int testTypeId = 0;
                int mappedTypeId = 0;

                //Initialize AutoInstrumentComments with an empty string.
                pendingTest.AutoInstrumentComments = string.Empty;

                //Lookup the test tech Id from the user table. Defect 251417 changed to DataRow, checks below are the same
                DataRow testTechRow = null;


                //lookup user only if necessary
                if (!lookedupUsers.ContainsKey(pendingTest.TestTechId))
                {
                    testTechRow = GetUserRecord(pendingTest.TestTechId, ref validationErrors);
                    if (testTechRow == null || StDbNullConvert.ToString(testTechRow[TABLE.VbecsUser.VbecsUserId]) == null)
                    {
                        validationErrors.Add(string.Format("Invalid User ID: {0} on Test: {1}.", pendingTest.TestTechId,
                            pendingTest.BloodTestTypeName));
                        continue;
                    }
                    lookedupUsers.Add(pendingTest.TestTechId, testTechRow);
                }
                else
                {
                    testTechRow = lookedupUsers[pendingTest.TestTechId];
                }

                //Assign the primary Test Tech ID.
                pendingTest.TestTechId = testTechRow[TABLE.VbecsUser.VbecsUserId].ToString();
                pendingTest.LastUpdateUser = testTechRow[TABLE.VbecsUser.VbecsUserId].ToString();

                //Defect 251417 - Save the user email - just take the first one
                userEmailForLockingFailure = userEmailForLockingFailure ?? StDbNullConvert.ToString(testTechRow[TABLE.VbecsUser.UserEmail]);

                //We check for interp so we can add a comment to this value.  
                //We only add comments to Interps that are not RhInterp.
                if (pendingTest.BloodTestTypeName.Contains("Interp") &&
                    !pendingTest.BloodTestTypeName.Contains("RhInterp"))
                {
                    pendingTest.AutoInstrumentComments = comment;
                }

                //Insert update task status row (only for specimen).
                if (!isBloodUnitType)
                {
                    if (pendingTests[0].TaskStatusCode !=
                        Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.PendingReview).ToString())
                        AddUpdateOrderedTestTaskStatus(ref updateOrderTaskStatusTable, pendingTest,
                            ref validationErrors);
                }                    

                //PT_114.01	HL7 Tests Accepted by VBECS
                //GetTestTypeMapping returns 0 for mappedTypeId if the table contains null.
                //   Test Type Name (HL7)     testTypeId (Test Type Name (VBECS))       mappedTypeId (Mapped to Interpretation)
                //        AntiE ->                     34 (Anti E) ->                              64 (AntiEInterp)
                //pendingTest.OrderableTestName="Antigen Typing" (OBR)
                //pendingTest.OrderableTestId=nnn (Looked up in VBECS, exist only for Specimen)
                //pendingTest.BloodTestTypeName=AntiE (OBX)
                if (!GetTestTypeMapping(pendingTest.OrderableTestName, pendingTest.BloodTestTypeName,
                    pendingTest.OrderableTestId, out testTypeId, out mappedTypeId, ref validationErrors))
                    continue;


                //Assign BloodTestTypeId prior testType management..
                pendingTest.BloodTestTypeId = testTypeId;

                //Here we handle antigen typing we have to add a row for antigen interpretation.
                //This this case we copy the pendingBloodUnitTest modify it to for antigen interpritation and add a row.
                //Next we modify pendingBloodUnitTest for antigen test type and add it in the normal way.
                if (mappedTypeId != 0 && isAntigenType)
                {
                    if (isBloodUnitType)
                    {
                        pendingTest.ResultSetGuid = Guid.NewGuid();
                        //Insert interp.
                        //Blood unit tests don't have orders so we don't have to update the OrderedTaskStatus.
                        if (!AntigenProcessing(mappedTypeId, pendingTest, comment,
                            ref pendingBloodUnitResultTable, ref updateOrderTaskStatusTable, ref validationErrors))
                            continue;
                    }
                    else
                    {
                        //Insert interp and Insert update task status row.
                        if (!AntigenProcessing(mappedTypeId, pendingTest, comment,
                            ref pendingSpecimenTestsResultTable, ref updateOrderTaskStatusTable, ref validationErrors))
                            continue;
                    }
                }
                    //Here we split the ABORhInterp into ABOInterp and RhInterp.
                    //This this case we copy the pendingBloodUnitTest modify it to for ABOInterp and add a row.
                    //Next we modify pendingBloodUnitTest for RhInterp and add it in the normal way.
                else if (mappedTypeId != 0 && pendingTest.BloodTestTypeName == "ABORhInterp")
                {
                    string interp;

                    //Split and Insert interp.
                    if (isBloodUnitType)
                    {
                        if (!AboRhIterpProcessing(testTypeId, comment, pendingTest,
                            out interp, ref pendingBloodUnitResultTable, ref updateOrderTaskStatusTable,
                            ref validationErrors))
                            continue;
                    }
                    else
                    {
                        if (!AboRhIterpProcessing(testTypeId, comment, pendingTest,
                            out interp, ref pendingSpecimenTestsResultTable, ref updateOrderTaskStatusTable,
                            ref validationErrors))
                            continue;
                    }
                    pendingTest.BloodTestTypeId = mappedTypeId;
                    pendingTest.TestResultText = interp;
                }

                if (pendingTest.BloodTestTypeName.Contains("Interp"))
                {
                    //Lookup interpretation 
                    //Pos -> P
                    if (bloodTestPanelName != BloodTestPanelName.XM)
                    {
                        pendingTest.TestResultId = InterpResultLookup(pendingTest, ref validationErrors);
                        if (pendingTest.AutoInstrumentComments == null)
                            pendingTest.AutoInstrumentComments = string.Empty;
                        if (string.IsNullOrEmpty(pendingTest.TestResultId))
                            continue;
                    }
                }
                else
                {
                    //Lookup result
                    //2 -> 2
                    pendingTest.TestResultId = ResultLookup(pendingTest, ref validationErrors);
                    if (string.IsNullOrEmpty(pendingTest.TestResultId))
                        continue;
                }

                if (isBloodUnitType)
                {
                    AddPendingUnitRow(ref pendingBloodUnitResultTable, (PendingBloodUnitTest) pendingTest);
                }
                else
                {
                    AddPendingSpecimenRow(ref pendingSpecimenTestsResultTable, (PendingSpecimenTest) pendingTest);
                }
            }

            #endregion

            //BR_114.04
            //A received order currently pending review must be rejected by the user in VBECS before replacement results can be received.
            if (isBloodUnitType)
            {
                PendingBloodUnitTest pendingTestRow = pendingTests.Any()
                    ? pendingTests.First() as PendingBloodUnitTest 
                    : new PendingBloodUnitTest();
                if (pendingTestRow == null)
                {
                    validationErrors.Add("There are no results to process for Blood Unit");
                    return validationErrors;
                }

                //BR_114.04
                //Check if test results already exist
                foreach (DataRow drTestResult in pendingBloodUnitResultTable.Rows)
                {
                    //BloodTestTypeId is coming originally from OBX-3, AntiB -> NN (Anti B)
                    if (drTestResult == null || drTestResult[TABLE.PendingBloodUnitTest.BloodTestTypeId] == null)
                    {
                        validationErrors.Add(string.Format(
                            "There are no results to process for Blood Unit: {0}.", pendingTestRow.BloodUnitId)); 
                        break;
                    }

                    bool pendingTestExists = PendingBloodUnitTestExists(pendingTestRow.BloodUnitId,
                        pendingTestRow.DivisionCode,
                        Convert.ToInt16(drTestResult[TABLE.PendingBloodUnitTest.BloodTestTypeId]));
                    if (pendingTestExists)
                    {
                        if(bloodTestPanelName == BloodTestPanelName.XM)
                        validationErrors.Add(string.Format(
                            "There is already a pending crossmatch test for Blood Unit ID: {0}.",
                            pendingTestRow.BloodUnitId));
                        else
                        {
                            validationErrors.Add(string.Format(
                                "There is already a pending blood unit test for Blood Unit ID: {0}.", 
                                pendingTestRow.BloodUnitId));
                            
                        }
                        break;
                    }
                }               
                //Lock check - Defect 251417
                if (validationErrors.Count == 0)
                {
                    if (LockManager.IsPendingBloodUnitTestLocked(pendingTestRow.BloodUnitGuid,
                                                                    pendingTestRow.BloodUnitId,
                                                                    pendingTestRow.OrderedUnitGuid,
                                                                    pendingTestRow.DivisionCode))
                    {
                        string lockErrorText = string.Format(StrRes.ValidMsg.Locking.PendingBloodUnitTestExists(pendingTestRow.BloodUnitId).ResString, pendingTestRow.BloodUnitId);
                        validationErrors.Add(lockErrorText);

                        SendLockingConflictEmail(userEmailForLockingFailure, lockErrorText);
                    }
                }
                
            }
            else
            {
                string otGuidArray = string.Empty;
                string strComma = string.Empty;
            
                PendingSpecimenTest pendingTestRow = pendingTests.Any() 
                    ? pendingTests.First() as PendingSpecimenTest 
                    : new PendingSpecimenTest();
                if (pendingTestRow == null)
                {
                    validationErrors.Add("There are no results to process for Specimen");
                    return validationErrors;
                }
                
                //BR_114.04
                //Check if test results already exist
                foreach (DataRow drTestResult in pendingSpecimenTestsResultTable.Rows)
                {
                    if (drTestResult == null || drTestResult[TABLE.PendingSpecimenTest.BloodTestTypeId] == null)
                    {
                        validationErrors.Add(string.Format(
                          "There are not specimen test rows to process for Specimen UID: {0} and Test Name: {1}.",
                          pendingTestRow.SpecimenUid, pendingTestRow.OrderName));
                        break;                      
                    }
                    //BloodTestTypeId is coming originally from OBX-3, AntiB -> NN (Anti B)
                    bool pendingTestExists = PendingSpecimenTestExists(pendingTestRow.SpecimenUid,
                        pendingTestRow.DivisionCode,
                        Convert.ToInt16(drTestResult[TABLE.PendingSpecimenTest.BloodTestTypeId]));
                    if (pendingTestExists)
                    {
                        validationErrors.Add(string.Format(
                            "There is already a pending specimen test for Specimen UID: {0} and Test Name: {1}.",
                            pendingTestRow.SpecimenUid, pendingTestRow.OrderName));
                        break;
                    }
                    otGuidArray += strComma + drTestResult[TABLE.PendingSpecimenTest.OrderedTestGuid].ToString();   //mooching off loop for locking
                    strComma = ",";
                }
                //Lock check - Defect 251417
                if (validationErrors.Count == 0)
                {
                    if (LockManager.IsPendingSpecimenTestLocked(pendingTestRow.PatientSpecimenGuid,
                                                                    otGuidArray,
                                                                    pendingTestRow.DivisionCode))
                    {
                        string lockErrorText = string.Format(StrRes.ValidMsg.Locking.PendingSpecimenTestExists(pendingTestRow.SpecimenUid).ResString,
                            pendingTestRow.SpecimenUid);
                        validationErrors.Add(lockErrorText);

                        SendLockingConflictEmail(userEmailForLockingFailure, lockErrorText);
                    }
                }
                
            }

            //Insert completed datatable (if no error).
            if (validationErrors.Count == 0)
            {
                if (isBloodUnitType)
                {
                    sprocArray.Add(STOREDPROC.HL7InsertPendingBloodUnitTest.StoredProcName);
                    dtArray.Add(pendingBloodUnitResultTable);
                    if (bloodTestPanelName == BloodTestPanelName.XM && updateOrderTaskStatusTable.Rows.Count > 0)
                    {
                        dtArray.Add(updateOrderTaskStatusTable);
                        sprocArray.Add(STOREDPROC.UpdateOrderedTestTaskStatus.StoredProcName);
                    }
                }
                else
                {
                    sprocArray.Add(STOREDPROC.HL7InsertPendingSpecimenTest.StoredProcName);
                    sprocArray.Add(STOREDPROC.UpdateOrderedTestTaskStatus.StoredProcName);
                    dtArray.Add(pendingSpecimenTestsResultTable);
                    dtArray.Add(updateOrderTaskStatusTable);
                }
            }
            return validationErrors;
        }

        /// <summary>
        /// Sends email if non-null arg, logs error on failure.
        /// Added for Defect 251417
        /// </summary>
        /// <param name="emailAddress">email address</param>
        /// <param name="messageBody">mail content</param>
        private static void SendLockingConflictEmail(string emailAddress, string messageBody)
        {
            if (emailAddress != null)
            {
                try
                {
                    new HL7MailMessage().SendMessage(emailAddress, PII                                    ", "Pending Results Controller Error", messageBody, "smtp.DNS   ");
                }
                catch
                {
                    //toss this up to the events/SCOM so we can yell at the dummies
                    _eventsLogger.Info(string.Format("Error trying to email <{0}> : {1}", emailAddress, messageBody));
                }
            }
        }

        /// <summary>
        /// Creates a DataTable for Blood Unit Inserts.
        /// </summary>
        /// <returns>A parameter table for pending blood unit inserts.</returns>
        public static DataTable GetInsertPendingBloodUnitTestSprocTable()
        {
            DataTable dt = new DataTable(VbecsTables.PendingBloodUnitTest.TableName);

            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.BloodUnitTestGuid, typeof(Guid));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.BloodUnitGuid, typeof(Guid));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.BloodTestTypeId, typeof(int));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.BloodUnitId, typeof(string));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.TestDate, typeof(DateTime));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.TestTechId, typeof(string));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.TestResultId, typeof(string));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.AutoInstrumentComments, typeof(string));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.OrderedTestGuid, typeof(Guid));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.OrderedComponentGuid, typeof(Guid));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.AutoInstrumentName, typeof(string));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.DivisionCode, typeof(string));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.PendingTestStatusId, typeof(int));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.UnitResultSetGuid, typeof(Guid));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.OrderedUnitGuid, typeof(Guid));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.LastUpdateUser, typeof(string));
            dt.Columns.Add(VbecsTables.PendingBloodUnitTest.LastUpdateFunctionId, typeof(int));

            return (dt);
        }

        /// <summary>
        /// Creates a DataTable for Patient Specimen Inserts.
        /// </summary>
        /// <returns>A parameter table for pending specimen inserts.</returns>
        public static DataTable GetInsertPendingPatientSpecimenTestSprocTable()
        {
            DataTable dt = new DataTable(VbecsTables.PendingSpecimenTest.TableName);

            dt.Columns.Add(VbecsTables.PendingSpecimenTest.OrderedTestGuid, typeof(Guid));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.PatientSpecimenGuid, typeof(Guid));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.BloodTestTypeId, typeof(int));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.TestDate, typeof(DateTime));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.TestTechId, typeof(string));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.TestResultId, typeof(string));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.AutoInstrumentName, typeof(string));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.AutoInstrumentComments, typeof(string));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.DivisionCode, typeof(string));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.PendingTestStatusId, typeof(int));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.SpecimenResultSetGuid, typeof(Guid));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.LastUpdateUser, typeof(string));
            dt.Columns.Add(VbecsTables.PendingSpecimenTest.LastUpdateFunctionId, typeof(int));
            return (dt);
        }

        /// <summary>
        /// Creates a DataTable for updating the OrderStatusCode.
        /// </summary>
        /// <returns>A parameter table updating the task status code in the OrderedTest table.</returns>
        public static DataTable GetOrderTaskStatusSprocTable()
        {
            DataTable dt = new DataTable(VbecsTables.PendingSpecimenTest.TableName); 
            dt.Columns.Add(VbecsTables.OrderedTest.OrderedTestGuid, typeof(Guid));
            dt.Columns.Add(VbecsTables.OrderedTest.RowVersion, typeof(Byte[]));
            dt.Columns.Add(VbecsTables.OrderedTest.TaskStatusCode, typeof(string));
            dt.Columns.Add(VbecsTables.OrderedTest.LastUpdateUser, typeof(string));
            dt.Columns.Add(VbecsTables.OrderedTest.LastUpdateFunctionId, typeof(int));
            return dt;

        }

        #region Private Classes


        /// <summary>
        /// Creates a DataTable for updating the OrderStatusCode.
        /// </summary>
        /// <returns>A parameter table updating the task status code in the OrderedTest table.</returns>
        public class PendingBloodUnitTest : PendingTest
        {
            /// <summary>
            /// Initialize.
            /// </summary>
            /// <param name="pendingTest"></param>
            public PendingBloodUnitTest(PendingBloodUnitTest pendingTest)
                : base(pendingTest)
            {
                PendingBloodUnitTestId = pendingTest.PendingBloodUnitTestId;
                BloodUnitTestGuid = pendingTest.BloodUnitTestGuid;
                BloodUnitGuid = pendingTest.BloodUnitGuid;
                BloodUnitId = pendingTest.BloodUnitId;
                OrderedComponentGuid = pendingTest.OrderedComponentGuid;
                OrderedUnitGuid = pendingTest.OrderedUnitGuid;
                BloodUnitStatusGuid = pendingTest.BloodUnitStatusGuid;
            }

            /// <summary>
            /// Copy constructor.
            /// </summary>
            public PendingBloodUnitTest()
            {
                PendingBloodUnitTestId = 0;
                BloodUnitTestGuid = Guid.Empty;
                BloodUnitGuid = Guid.Empty;
                BloodUnitId = string.Empty;
                OrderedComponentGuid = Guid.Empty;
                OrderedUnitGuid = Guid.Empty;
                BloodUnitStatusGuid = Guid.Empty;
            }

            /// <summary>
            /// PendingBloodUnitTestId
            /// </summary>
            public int PendingBloodUnitTestId { get; set; }
            /// <summary>
            /// BloodUnitTestGuid
            /// </summary>
            public Guid BloodUnitTestGuid { get; set; }
            /// <summary>
            /// BloodUnitGuid
            /// </summary>
            public Guid BloodUnitGuid { get; set; }
            /// <summary>
            /// BloodUnitId
            /// </summary>
            public string BloodUnitId { get; set; }
            /// <summary>
            /// OrderedComponentGuid
            /// </summary>
            public Guid OrderedComponentGuid { get; set; }
            /// <summary>
            /// OrderedUnitGuid
            /// </summary>
            public Guid OrderedUnitGuid { get; set; }
            /// <summary>
            /// BloodUnitStatusGuid
            /// </summary>
            public Guid BloodUnitStatusGuid { get; set; }
        }


        /// <summary>
        /// Creates a DataTable for updating the OrderStatusCode.
        /// </summary>
        /// <returns>A parameter table updating the task status code in the OrderedTest table.</returns>
        public class PendingSpecimenTest : PendingTest
        {
            /// <summary>
            /// Initialize.
            /// </summary>
            public PendingSpecimenTest()
            {
                SpecimenTestGuid = Guid.Empty;
                OrderName = string.Empty;
                SpecimenUid = string.Empty;
            }

            /// <summary>
            /// Copy constructor.
            /// </summary>
            /// <param name="pendingTest"></param>
            public PendingSpecimenTest(PendingSpecimenTest pendingTest)
                : base(pendingTest)
            {
                
                SpecimenTestGuid = pendingTest.SpecimenTestGuid;
                OrderName = pendingTest.OrderName;
                SpecimenUid = pendingTest.SpecimenUid;
            }

            /// <summary>
            /// SpecimenTestGuid
            /// </summary>
            public Guid SpecimenTestGuid { get; set; }
            /// <summary>
            /// OrderName
            /// </summary>
            public string OrderName { get; set; }
            /// <summary>
            /// SpecimenUid
            /// </summary>
            public string SpecimenUid { get; set; }
        }


        /// <summary>
        /// GetPendingBloodUnitTest Data Access Object. Used to convert data rows from spGetPendingBloodUnitTests
        /// to a strongly typed data access object.
        /// </summary>
        public class PendingTest 
        {

            /// <summary>
            /// Initialize.
            /// </summary>
            /// <exception cref="NotImplementedException"></exception>
            public PendingTest()
            {
                BloodTestTypeId = 0;
                BloodTestTypeName = string.Empty;
                TestTechId = string.Empty;
                TestResultId = string.Empty;
                TestTechUserName = string.Empty;
                TestResultText = string.Empty;
                AutoInstrumentComments = string.Empty;
                AutoInstrumentName = string.Empty;
                DivisionCode = string.Empty;
                PendingTestStatusId = 0;
                ResultSetGuid = Guid.Empty;
                OrderableTestId = 0;
                OrderableTestName = string.Empty;
                RowVersion = null;
                LastUpdateUser = string.Empty;
                OrderedTestGuid = Guid.Empty;
                PatientSpecimenGuid = Guid.Empty;
            }

            /// <summary>
            /// Copy constructor.
            /// </summary>
            /// <param name="pendingTest"></param>
            public PendingTest(PendingTest pendingTest)
            {

                BloodTestTypeId = pendingTest.BloodTestTypeId;
                BloodTestTypeName = pendingTest.BloodTestTypeName;
                TestTechId = pendingTest.TestTechId;
                TestDate = pendingTest.TestDate;
                TestResultId = pendingTest.TestResultId;
                TestTechUserName = pendingTest.TestTechUserName;
                TestResultText = pendingTest.TestResultText;
                AutoInstrumentComments = pendingTest.AutoInstrumentComments;
                AutoInstrumentName = pendingTest.AutoInstrumentName;
                DivisionCode = pendingTest.DivisionCode;
                PendingTestStatusId = pendingTest.PendingTestStatusId;
                ResultSetGuid = pendingTest.ResultSetGuid;
                OrderableTestId = pendingTest.OrderableTestId;
                OrderableTestName = pendingTest.OrderableTestName;
                RowVersion = pendingTest.RowVersion;
                LastUpdateUser = pendingTest.LastUpdateUser;
                OrderedTestGuid = pendingTest.OrderedTestGuid;
                PatientSpecimenGuid = pendingTest.PatientSpecimenGuid;
            }

            /// <summary>
            ///  BloodTestTypeId
            /// </summary>
            public int BloodTestTypeId { get; set; }
            /// <summary>
            ///  BloodTestTypeName
            /// </summary>
            public string BloodTestTypeName { get; set; }
            /// <summary>
            ///  TestDate
            /// </summary>
            public DateTime TestDate { get; set; }
            /// <summary>
            ///  TestTechId
            /// </summary>
            public string TestTechId { get; set; }
            /// <summary>
            ///  TestResultId
            /// </summary>
            public string TestResultId { get; set; }
            /// <summary>
            ///  TestTechUserName
            /// </summary>
            public string TestTechUserName { get; set; }
            /// <summary>
            ///  TestResultText
            /// </summary>
            public string TestResultText { get; set; }
            /// <summary>
            ///  AutoInstrumentComments
            /// </summary>
            public string AutoInstrumentComments { get; set; }
            /// <summary>
            ///  AutoInstrumentName
            /// </summary>
            public string AutoInstrumentName { get; set; }
            /// <summary>
            ///  DivisionCode
            /// </summary>
            public string DivisionCode { get; set; }
            /// <summary>
            ///  PendingTestStatusId
            /// </summary>
            public byte PendingTestStatusId { get; set; }
            /// <summary>
            ///  ResultSetGuid
            /// </summary>
            public Guid ResultSetGuid { get; set; }
            /// <summary>
            ///  OrderableTestId
            /// </summary>
            public int OrderableTestId { get; set; }
            /// <summary>
            ///  OrderableTestName
            /// </summary>
            public string OrderableTestName { get; set; }
            /// <summary>
            ///  RowVersion
            /// </summary>
            public byte[] RowVersion { get; set; }
            /// <summary>
            ///  LastUpdateUser
            /// </summary>
            public string LastUpdateUser { get; set; }
            /// <summary>
            /// OrderedTestGuid
            /// </summary>
            public Guid OrderedTestGuid { get; set; }
            /// <summary>
            /// PatientSpecimenGuid
            /// </summary>
            public Guid PatientSpecimenGuid { get; set; }
            /// <summary>
            ///  TaskStatusCode
            /// </summary>
            public string TaskStatusCode { get; set; }

        }



        #endregion


    }
}
